/**
 * jobqueue.c
 *
 * Дана програма моделює роботу багатопотокового паралельного сервера без
 * зворотного зв'язку між серверними і клієнтським потоками (але якщо
 * клієнтський потік представляє зовнішніх клієнтів, може мати місце
 * зворотний зв'язок між серверними потоками і зовнішніми клієнтами; для
 * цього клієнтський потік в структуру, яка описує завдання для серверного
 * потоку, повинен включати інформацію про спосіб зв'язку з зовнішнім
 * клієнтом). Використовується фіксована кількість завчасно створених
 * серверних потоків. Один клієнтський потік (який, можливо, представляє
 * зовнішніх клієнтів) періодично виробляє завдання (jobs) для серверних
 * потоків, які розміщує в черзі завдань (jobqueue). Серверні потоки
 * витягують завдання з черги завдань і обробляють їх. Клієнтський потік,
 * помістивши завдання в чергу, продовжує роботу (наприклад, чекає запиту
 * від нового зовнішнього клієнта). Програма виводить повідомлення про
 * розміщення в черзі чергового завдання і результат обробки кожного
 * завдання. Черга завдань реалізована як зв'язаний список структур.
 * Доступ до неї регулюється за допомогою  м'ютекса (job_queue_mutex)
 * і умовної змінної (job_queue_cond). 
 */
 
#include <assert.h>
#include <pthread.h>
#include <semaphore.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* Кількість серверних потоків */
enum { NSERVERS = 5 };

/* Структура, яка представляє завдання */
struct job 
{
        /* ... */
        struct job *next;
};

/* Покажчики на початок та кінець черги завдань */
struct job *job_queue_head, *job_queue_tail;

pthread_mutex_t job_queue_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t job_queue_cond;

int init_job_queue();
void client(void), *server(void *);


int main(int argc, char **argv)
{
        pthread_t server_threads[NSERVERS];
        int server_args[NSERVERS];
        int i, rval, terrno;

        /* Виконує ініціалізацію черги завдань. */
        if (init_job_queue() != 0)
                exit(EXIT_FAILURE);
        /* Створює серверні потоки. */
        for (i = 0; i < NSERVERS; i++) {
                server_args[i] = i;
                terrno = pthread_create(&server_threads[i], NULL, server,
                                                        &server_args[i]);
                if (terrno != 0) {
                        fprintf(stderr, "Error creating thread: %s\n",
                                                        strerror(terrno));
                        exit(EXIT_FAILURE);
                }
        }

        /* Заходить в основний робочий цикл. */
        client();

        /* Чекає завершення серверних потоків. */
        for (i = 0; i < NSERVERS; i++) {
                rval = pthread_join(server_threads[i], NULL);
                assert(rval == 0);
        }	

        exit(EXIT_SUCCESS);
}

/**
 * Розміщує структуру для нового завдання і виконує її ініціалізацію.
 * Повертає: покажчик на створену структуру у випадку успіху, NULL - 
 * у випадку невдачі.
 */
struct job *create_job()
{
        struct job *new_job;

        new_job = malloc(sizeof(*new_job));
        if (new_job == NULL) {
                fprintf(stderr, "Not enough memory to create new job\n");
                return NULL;
        }
        /* ... */
        new_job->next = NULL;
        return new_job;
}

/**
 * Звільняє пам'ять, виділену для завдання.
 */
void free_job(struct job *job)
{
        assert(job != NULL);
        free(job);
}

/**
 * Виконує ініціалізацію черги завдань.
 * Повертає: 0 у випадку успіху, інше значення у випадку невдачі.
 */
int init_job_queue()
{
        int rval; 

        job_queue_head = NULL;
        job_queue_tail = NULL;
        rval = pthread_cond_init(&job_queue_cond, NULL);
        if (rval != 0) {
                fprintf(stderr, "Error initializing conditional variable:"
                                                " %s\n", strerror(rval));
                return 1;
        }
        return 0;
}

/**
 * Додає завдання до (хвоста) черги завдань.
 * Аргументи: new_job - покажчик на структуру із новим завданням.
 */
void enqueue_job(struct job *new_job)
{
        int rval;

        assert(new_job != NULL);

        rval = pthread_mutex_lock(&job_queue_mutex);
        assert(rval == 0);

        if (job_queue_tail == NULL) {
                assert(job_queue_head == NULL);
                job_queue_head = new_job;
        } else
                job_queue_tail->next = new_job;
        job_queue_tail = new_job;
        assert(job_queue_head != NULL);

        printf("A job was requested\n");

        /* Будить один із серверних потоків, що, можливо, чекають появи
           в черзі нового завдання. */
        rval = pthread_cond_signal(&job_queue_cond);
        assert(rval == 0);

        rval = pthread_mutex_unlock(&job_queue_mutex);
        assert(rval == 0);

        return;
}

/**
 * Вилучає завдання з (голови) черги завдань.
 * Повертає покажчик на структуру з вилученим із черги завданням.
 */
struct job *dequeue_job()
{
        struct job *next_job;
        int rval;
    
        rval = pthread_mutex_lock(&job_queue_mutex);
        assert(rval == 0);

        /* Якщо черга порожня, чекає, доки в ній з'явиться завдання. */
        while (job_queue_head == NULL)
                rval = pthread_cond_wait(&job_queue_cond,
                                                &job_queue_mutex);
        assert(rval == 0);

        next_job = job_queue_head;
        job_queue_head = job_queue_head->next;
        if (job_queue_head == NULL)
                job_queue_tail = NULL;

        rval = pthread_mutex_unlock(&job_queue_mutex);
        assert(rval == 0);

        return next_job;
}

/**
 * Основний робчий цикл клієнтського потоку.
 */
void client()
{
        for (;;) {
                struct job *new_job;

                /* Виконує якусь роботу (можливо, взаємодіє з зовнішнім
                   клієнтом)... */

                /* Створює нове завдання. */
                new_job = create_job();
                if (new_job == NULL) 
                        continue;
                /* Додає завдання до черги. */
                enqueue_job(new_job);

                /* Виконує якусь іншу роботу... */

                /* Це тут тільки для того, щоб уповільнити роботу
                   клієнтського потоку. */
                sleep(1); 
        }
        return;
}

/**
 * Виконує обробку завдання.
 * Аргументи: job - покажчик на структуру з завданням, яке треба обробити.
 * Повертає: 0 у випадку успіху, інше значення у випадку невдачі.
 */
int process_job(struct job *job)
{
        assert(job != NULL);
        /* ... */
        return 0;
}

/**
 * Головна функція серверного потоку.
 * Аргументи: arg - покажчик на змінну типу int, яка зберігає номер
 * серверного потоку.
 */
void *server(void *arg)
{
        int number;             /* Номер серверного потоку */

        number = *((int *) arg);
        for (;;) {
                struct job *next_job;

                /* Вилучає з черги нове завдання. */
                next_job = dequeue_job();
                /* Обробляє завдання. */
                if (process_job(next_job) != 0)
                        fprintf(stderr, "Error processing job\n");
                else
                        printf("A job was processed by server %d\n",
                                                                number);
                /* Звільняє пам'ять, виділену для завдання. */
                free_job(next_job);
        }
        return NULL;
}
